---
title: "Gestion des données avec les volumes"
course: "docker"
slug: "6-gestion-des-donnees-avec-les-volumes"
order: 6
createdAt: 2025-09-24
updatedAt: 2025-09-24
---

Par défaut, tout ce qui se passe à l'intérieur d'un conteneur disparaît quand il est supprimé. Pour préserver les données, Docker propose les **volumes**.

---

## Qu'est-ce qu'un volume ?

Un **volume Docker** est un espace de stockage persistant utilisé par un ou plusieurs conteneurs. Il est indépendant du cycle de vie des conteneurs.

> [!info] Pourquoi utiliser des volumes ?
> Idéal pour bases de données, logs, fichiers générés, configuration, etc.

---

## Deux types principaux

|Type|Description|Cas d'usage|
|---|---|---|
|**Named volume**|Volume géré par Docker (stocké dans `/var/lib/docker/volumes`)|Production, bases de données|
|**Bind mount**|Monte un dossier local dans le conteneur|Développement, fichiers de config|

---

## Named Volumes (recommandé pour la production)

### Créer un volume

```bash
docker volume create mon_volume
```

### Utiliser un volume avec un conteneur

```bash
docker run -d -v mon_volume:/data --name mon_conteneur busybox
```

**Avec la syntaxe `--mount` (recommandée) :**

```bash
docker run -d --name mon_conteneur \
  --mount source=mon_volume,target=/data \
  busybox
```

### Exemple concret : Base de données PostgreSQL

```bash
# Créer un volume pour les données
docker volume create postgres_data

# Lancer PostgreSQL avec le volume
docker run -d \
  --name postgres_db \
  --mount source=postgres_data,target=/var/lib/postgresql/data \
  -e POSTGRES_PASSWORD=motdepasse \
  -p 5432:5432 \
  postgres:15
```

**Avantages des named volumes :**

- Gérés entièrement par Docker
- Sécurisés et isolés
- Portables entre environnements
- Sauvegarde facilitée
- Partageable entre conteneurs

---

## Bind Mounts (pour le développement)

### Exemple simple

```bash
docker run -v $(pwd)/logs:/app/logs mon_app
```

**Avec `--mount` :**

```bash
docker run --mount type=bind,source="$(pwd)"/logs,target=/app/logs mon_app
```

### Exemple concret : Développement d'une app web

```bash
# Monter le code source pour le développement
docker run -d \
  --name dev_app \
  --mount type=bind,source="$(pwd)",target=/app \
  -p 3000:3000 \
  node:18 \
  npm run dev
```

**Avantages des bind mounts :**

- Modifications en temps réel
- Accès direct aux fichiers depuis l'hôte
- Parfait pour le développement
- Partage de fichiers de configuration

> [!warning] Sécurité
> Les bind mounts donnent accès au système de fichiers de l'hôte. Utilisez avec précaution en production.

---

## Gestion des volumes

### Lister les volumes

```bash
docker volume ls
```

### Inspecter un volume

```bash
docker volume inspect mon_volume
```

**Informations retournées :**

```json
[
    {
        "CreatedAt": "2025-01-15T10:30:45Z",
        "Driver": "local",
        "Mountpoint": "/var/lib/docker/volumes/mon_volume/_data",
        "Name": "mon_volume"
    }
]
```

### Supprimer un volume

```bash
docker volume rm mon_volume
```

> [!warning] Attention
> Docker ne supprime pas automatiquement les volumes quand un conteneur est supprimé !

---

## Partager des volumes entre conteneurs

### Exemple : App web + Base de données

```bash
# Créer un volume partagé
docker volume create app_data

# Conteneur de base de données
docker run -d \
  --name db \
  --mount source=app_data,target=/var/lib/mysql \
  -e MYSQL_ROOT_PASSWORD=root \
  mysql:8

# Conteneur d'application
docker run -d \
  --name app \
  --mount source=app_data,target=/app/data \
  -p 8080:8080 \
  mon_app
```

---

## Volumes en lecture seule

```bash
docker run -d \
  --mount source=config_vol,target=/etc/config,readonly \
  mon_app
```

**Avec `-v` :**

```bash
docker run -d -v config_vol:/etc/config:ro mon_app
```

---

## Nettoyage des volumes

### Supprimer les volumes non utilisés

```bash
docker volume prune
```

**Cette commande supprime :**

- Tous les volumes qui ne sont attachés à aucun conteneur
- Attention : irréversible !

### Forcer la suppression

```bash
docker volume prune -f
```

---

## Bonnes pratiques

### Pour le développement

- **Utilisez des bind mounts** pour le code source
- **Montez les fichiers de config** depuis l'hôte
- **Utilisez des volumes nommés** pour les données

```bash
# Exemple de développement optimal
docker run -d \
  --name dev_container \
  --mount type=bind,source="$(pwd)",target=/app \
  --mount type=bind,source="$(pwd)"/config,target=/etc/app \
  --mount source=dev_data,target=/app/data \
  -p 8000:8000 \
  mon_app
```

### Pour la production

- **Toujours utiliser des named volumes** pour les données critiques
- **Bind mounts uniquement** pour les configs read-only
- **Backups réguliers** des volumes importants

```bash
# Exemple de production
docker run -d \
  --name prod_app \
  --mount source=prod_data,target=/app/data \
  --mount type=bind,source=/etc/app-config,target=/etc/app,readonly \
  --restart unless-stopped \
  mon_app
```

### Sauvegarde d'un volume

```bash
# Créer une sauvegarde
docker run --rm \
  --mount source=mon_volume,target=/data \
  --mount type=bind,source="$(pwd)",target=/backup \
  ubuntu tar czf /backup/backup.tar.gz -C /data .

# Restaurer une sauvegarde
docker run --rm \
  --mount source=nouveau_volume,target=/data \
  --mount type=bind,source="$(pwd)",target=/backup \
  ubuntu tar xzf /backup/backup.tar.gz -C /data
```

---

## Comparaison --mount vs -v

|Aspect|`--mount`|`-v`|
|---|---|---|
|**Lisibilité**|Plus explicite|Plus concis|
|**Flexibilité**|Plus d'options|Limité|
|**Erreurs**|Messages clairs|Moins informatif|
|**Production**|Recommandé|Acceptable|

**Exemple équivalent :**

```bash
# Avec --mount (recommandé)
docker run --mount type=volume,source=data,target=/app/data mon_app

# Avec -v
docker run -v data:/app/data mon_app
```

---

## Diagnostic et résolution de problèmes

### Vérifier l'utilisation d'un volume

```bash
docker ps --filter volume=mon_volume
```

### Voir les volumes orphelins

```bash
docker volume ls -f dangling=true
```

### Localiser les données d'un volume

```bash
docker volume inspect mon_volume | grep Mountpoint
```

---

## À retenir

- **Named volumes** : production, données critiques, gérés par Docker
- **Bind mounts** : développement, configs, accès temps réel
- **Toujours nettoyer** les volumes non utilisés
- **Sauvegarder** les volumes importants
- **Utiliser `--mount`** pour plus de clarté
- **Volumes en lecture seule** pour les configurations
- **Attention aux permissions** sur les bind mounts

---

## Liens vers les autres parties

- [[7. Docker Compose (intro simple)]]

---

## Tags

#docker #volumes #persistance #bind-mounts #named-volumes #données #stockage